<?php

namespace App\Http\Controllers\ControlPanel;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreTeamRequest;
use App\Http\Requests\UpdateTeamRequest;
use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class TeamController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $teams = Team::all();
            return datatables()->of($teams)
                ->editColumn('name', function (Team $team) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $team->name_ar;
                    } else {
                        return $team->name;
                    }
                })
                ->editColumn('position', function (Team $team) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $team->position_ar;
                    } else {
                        return $team->position;
                    }
                })
                ->editColumn('image', function (Team $team) {
                    return '<img src="' . asset('storage/' . $team->image) . '" width="50" alt="' . $team->name . '">';
                })
                ->addColumn('actions', function (Team $team) {
                    $delete = '<a href="#" class="btn btn-danger btn-sm" data-toggle= "modal" data-target= "#modals-delete-' . $team->id . '">' .
                        __('حذف').'</a>';
                    $edit = ' <a href="' . route('teams.edit', $team->id) . '" class="btn btn-sm btn-primary">'.__('تعديل').'</a>';

                    return $delete . $edit;

                })
                ->rawColumns(['actions', 'image'])
                ->make(true);
        }
        $teams = Team::all();

        return view('control-panel.teams.index',[
            'teams' => $teams,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('control-panel.teams.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreTeamRequest $request)
    {
        $data = $request->validated();

        $image = null;

        if($request->hasFile('image') && $request->file('image')->isValid())
        {
            $image = $request->file('image')->store('team','public');
        }

        $data['image'] = $image;

        $team = Team::create($data);

        return redirect()->route('teams.index')->with('success',__('Team Member ').$team->name.__(' Created Done!'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Team $team)
    {
        return view('control-panel.teams.edit',[
            'team' => $team,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateTeamRequest $request, Team $team)
    {
        $data = $request->validated();

        $image = $team->image;

        if($request->hasFile('image') && $request->file('image')->isValid())
        {
            Storage::disk('public')->delete($image);
            $image = $request->file('image')->store('team','public');
            $data['image'] = $image;
        }

        $team->update($data);

        return redirect()->route('teams.index')->with('success',__('Team Member Updated Done!'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Team $team)
    {
        $team->delete();
        Storage::disk('public')->delete($team->image);
        return redirect()->route('teams.index')->with('success',__('Team Member Deleted Done!'));
    }
}
