<?php

namespace App\Http\Controllers\ControlPanel;

use App\Http\Controllers\Controller;
use App\Models\StudyCase;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use Throwable;

class StudyCasesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $study_cases = StudyCase::all();
            return datatables()->of($study_cases)
                ->editColumn('title', function (StudyCase $study_case) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $study_case->title_ar;
                    } else {
                        return $study_case->title;
                    }
                })
                ->editColumn('sub_title', function (StudyCase $study_case) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $study_case->sub_title_ar;
                    } else {
                        return $study_case->sub_title;
                    }
                })
                ->editColumn('main_image', function (StudyCase $study_case) {
                    return '<img src="' . asset('storage/' . $study_case->main_image) . '" width="50" alt="' . $study_case->title . '">';
                })
                ->addColumn('actions', function (StudyCase $study_case) {
                    $delete = '<a href="#" class="btn btn-danger btn-sm" data-toggle= "modal" data-target= "#modals-delete-' . $study_case->id . '">' .
                        __('حذف').'</a>';
                    $edit = ' <a href="' . route('study-cases.edit', $study_case->id) . '" class="btn btn-sm btn-primary">'.__('تعديل').'</a>';

                    return $delete . $edit;

                })
                ->rawColumns(['actions', 'main_image'])
                ->make(true);
        }
        $study_cases = StudyCase::all();
        return view('control-panel.study_cases.index', [
            'study_cases' => $study_cases,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('control-panel.study_cases.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'required |max:150',
            'title_ar' => 'required |max:150',
            'study_category_id' => 'required|exists:study_categories,id',
            'shot_desc' => 'required',
            'shot_desc_ar' => 'required',
            'main_image' => 'required |image |mimes:png,jpg',
            'sub_title' => 'nullable |string',
            'sub_title_ar' => 'nullable |string',
            'gallery' => 'nullable |array ',
            'description' => 'nullable',
            'description_ar' => 'nullable',
        ]);

        $image = null;
        DB::beginTransaction();

        try {
            if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
                $image = $request->file('main_image')->store('blogs', 'public');
            }

            $data['main_image'] = $image;

            $study_case = StudyCase::create($data);

            if ($request->hasFile('gallery')) {
                foreach ($request->file('gallery') as $file) {
                    $image = $file->store('study_cases', 'public');
                    $study_case->images()->create([
                        'image' => $image,
                    ]);
                }
            }

            DB::commit();

        } catch (Throwable $e) {
            DB::rollBack();
            return redirect()->route('study-cases.index')
                ->with('error', 'Operation failed');
        }

        return redirect()->route('study-cases.index')->with('success', __(' Created Done!'));

    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit(StudyCase $study_case)
    {
        return view('control-panel.study_cases.edit', [
            'study_case' => $study_case,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, StudyCase $study_case)
    {
        $data = $request->validate([
            'title' => 'required |max:150',
            'title_ar' => 'required |max:150',
            'shot_desc' => 'required',
            'study_category_id' => 'required|exists:study_categories,id',
            'shot_desc_ar' => 'required',
            'main_image' => 'image |mimes:png,jpg',
            'sub_title' => 'nullable |string',
            'sub_title_ar' => 'nullable |string',
            'gallery' => 'nullable |array ',
            'description' => 'nullable',
            'description_ar' => 'nullable',
        ]);

        $image = $study_case->main_image;

        DB::beginTransaction();

        try {

            if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
                Storage::disk('public')->delete($image);
                $image = $request->file('main_image')->store('study_cases', 'public');
                $data['main_image'] = $image;
            }


            foreach ($study_case->images as $gallery) {
                if ($request->has("check_" . $gallery->id) == 1) {
                    Storage::disk('public')->delete($gallery->image);
                    $gallery->delete();
                }
            }


            $study_case->update($data);

            if ($request->hasFile('gallery')) {
                foreach ($request->file('gallery') as $file) {
                    $gallery_image = $file->store('study_cases', 'public');
                    $study_case->images()->create([
                        'image' => $gallery_image,
                    ]);
                }
            }
            DB::commit();

        } catch (Throwable $e) {
            DB::rollBack();
            return redirect()->route('study-cases.index')
                ->with('error', 'Operation failed');
        }

        return redirect()->route('study-cases.index')->with('success', __(' Updated Done!'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(StudyCase $study_case)
    {
        foreach ($study_case->images as $gallery) {
            Storage::disk('public')->delete($gallery->image);
            $gallery->delete();
        }

        Storage::disk('public')->delete($study_case->main_image);
        $study_case->delete();
        return redirect()->route('study-cases.index')->with('success', __('Deleted Done!'));
    }

}
