<?php

namespace App\Http\Controllers\ControlPanel;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreProductsRequest;
use App\Http\Requests\UpdateProductsRequest;
use App\Models\Product;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class ProductsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $products = Product::all();
            return datatables()->of($products)
                ->editColumn('main_image', function (Product $product) {
                    return '<img src="' . asset('storage/' . $product->main_image) . '" width="50" alt="' . $product->name . '">';
                })
                ->editColumn('name', function (Product $product) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $product->name_ar;
                    } else {
                        return $product->name_en;
                    }
                })
                ->editColumn('id', function (Product $product) {
                    return $product->code ?? $product->id;
                })
                ->editColumn('created_at', function (Product $product) {
                    return Carbon::parse($product->created_at)->format('y-m-d g:i A');
                })
                ->addColumn('category', function (Product $product) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $product->subCategory->name ?? 'There is no';
                    } else {
                        return $product->subCategory->name_en ?? 'There is no';
                    }
                })
                ->addColumn('actions', function (Product $product) {
                    $delete = '<a href="javascript:void(0);" class="btn btn-danger btn-sm" data-toggle= "modal" data-target= "#modals-delete-' . $product->id . '">'
                        . __('حذف') . '</a>';
                    $edit = ' <a href="'.route('products.edit',$product->id).'" class="btn btn-sm btn-primary">' . __('تعديل') . '</a>';
                    $create = ' <a href="'.route('benefits.index').'?productID=' . $product->id . '" class="btn btn-sm btn-success">' . __('عرض واضافة ميزات') . '</a>';

                    return $delete . $edit . $create;
                })
                ->rawColumns(['actions', 'main_image'])
                ->make(true);
        }

        $products = Product::all();
        return view('control-panel.products.index', [
            'products' => $products
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('control-panel.products.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreProductsRequest $request)
    {
        $data = $request->validated();
        if (!str_contains($request->link, 'https://www.youtube.com/watch?v=')) {
            return redirect()->back()->with('error', __('يجب ان يكون الرابط رابط يوتيوب'));
        }
        $image = null;
        $file = null;
        if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
            $image = $request->file('main_image')->store('products', 'public');
        }
        if ($request->hasFile('file') && $request->file('file')->isValid()) {
            $file = $request->file('file')->store('products', 'public');
        }
        $data['main_image'] = $image;
        $data['file'] = $file;

        $product = Product::create($data);

        if ($request->hasFile('gallery')) {
            foreach ($request->file('gallery') as $file) {
                $image = $file->store('products', 'public');
                $product->images()->create([
                    'image' => $image,
                ]);
            }
        }

        return redirect()->route('products.index')->with('success',__(' Created Done!'));
    }



    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Product $product)
    {
        return view('control-panel.products.edit',[
            'product' => $product,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateProductsRequest $request, Product $product)
    {
        $data = $request->validated();
        if (!str_contains($request->link, 'https://www.youtube.com/watch?v=')) {
            return redirect()->back()->with('error', __('يجب ان يكون الرابط رابط يوتيوب'));
        }

        $image = $product->main_image;
        $file = $product->file;
        if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
            $image = $request->file('main_image')->store('products', 'public');
        }
        if ($request->hasFile('file') && $request->file('file')->isValid()) {
            $file = $request->file('file')->store('products', 'public');
        }
        $data['main_image'] = $image;
        $data['file'] = $file;

        foreach ($product->images as $gallery) {
            if ($request->has("check_" . $gallery->id) == 1) {
                Storage::disk('public')->delete($gallery->image);
                $gallery->delete();
            }
        }

        $product->update($data);

        if ($request->hasFile('gallery')) {
            foreach ($request->file('gallery') as $file) {
                $image = $file->store('products', 'public');
                $product->images()->create([
                    'image' => $image,
                ]);
            }
        }

        return redirect()->route('products.index')->with('success',__(' Updated Done!'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Product $product)
    {
        foreach ($product->images as $gallery) {
            Storage::disk('public')->delete($gallery->image);
            $gallery->delete();
        }

        Storage::disk('public')->delete($product->main_image);
        $product->delete();
        return redirect()->route('products.index')->with('success', __('Deleted Done!'));
    }
}
