<?php

namespace App\Http\Controllers\ControlPanel;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePageRequest;
use App\Http\Requests\UpdatePageRequest;
use App\Models\Menu;
use App\Models\Page;
use App\Models\PageTag;
use App\Models\SubMenu;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use Throwable;

class PageController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $pages = Page::all();
            return datatables()->of($pages)
                ->editColumn('main_image', function (Page $page) {
                    return '<img src="' . asset('storage/' . $page->main_image) . '" width="50" alt="' . $page->title . '">';
                })
                ->editColumn('title', function (Page $page) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $page->title_ar;
                    } else {
                        return $page->title;
                    }
                })
                ->editColumn('sub_title', function (Page $page) {
                    if(LaravelLocalization::getCurrentLocale() == 'ar') {
                        return $page->sub_title_ar;
                    } else {
                        return $page->sub_title;
                    }
                })
                ->addColumn('actions', function (Page $page) {

                    $buttons = '
                    <div class="btn-group">
                        <button class="btn btn-outline-primary dropdown-toggle waves-effect" type="button" id="dropdownMenuButton{{ $item->id }}" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                            العمليات
                        </button>
                        <div class="dropdown-menu" aria-labelledby="dropdownMenuButton{{ $item->id }}">
                            <a href="' . route('pages.show', $page->id) . '" class="dropdown-item">
                                <svg viewBox="0 0 24 24" width="15" height="15" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round" class="css-i6dzq1"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>
                                <span style="margin: rigth 10px;" >'.__('عرض').'</span></a>
                            <a href="' . route('pages.edit', $page->id) . '" class="dropdown-item">
                                <svg viewBox="0 0 24 24" width="15" height="15" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round" class="css-i6dzq1"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path></svg>
                                <span style="margin: rigth 10px;" >'.__('تعديل').'</span></a>
                            <a href="#" class="dropdown-item" data-toggle= "modal" data-target= "#modals-delete-{{ $item->id }}">
                                    <svg viewBox="0 0 24 24" width="15" height="15" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round" class="css-i6dzq1"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path><line x1="10" y1="11" x2="10" y2="17"></line><line x1="14" y1="11" x2="14" y2="17"></line></svg>
                                    <span style="margin: rigth 10px;" >'.__('حذف').'</span></a>
                            <a href="' . route('features.index', $page->id) . '" class="dropdown-item">
                                <svg viewBox="0 0 24 24" width="15" height="15" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round" class="css-i6dzq1"><path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path></svg>
                                <span style="margin: rigth 10px;">'.__('المزايا').'</span></a>
                            <a href="' . route('themes.index', $page->id) . '" class="dropdown-item">
                            <svg viewBox="0 0 24 24" width="15" height="15" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round" class="css-i6dzq1"><path d="M20.24 12.24a6 6 0 0 0-8.49-8.49L5 10.5V19h8.5z"></path><line x1="16" y1="8" x2="2" y2="22"></line><line x1="17.5" y1="15" x2="9" y2="15"></line></svg>
                            <span style="margin: rigth 10px;">'.__('الموضوعات').'</span></a>
                        </div>
                    </div>

                    ';

                    return $buttons;

                })
                ->rawColumns(['actions', 'main_image'])
                ->make(true);
        }
        $pages = Page::all();
        return view('control-panel.pages.index', [
            'pages' => $pages,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
//        dd(Menu::has('subMenus')->get());
        return view('control-panel.pages.create', ['menus' => Menu::where('id','<>',2)->get()]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(StorePageRequest $request)
    {
        $data = $request->validated();

        // dd($data);
        $image = null;
        $header_image = null;

//        dd($data);
        DB::beginTransaction();

        try {
            if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
                $image = $request->file('main_image')->store('pages', 'public');
            }

            if ($request->hasFile('header_image') && $request->file('header_image')->isValid()) {
                $header_image = $request->file('header_image')->store('pages', 'public');
            }

            $data['header_image'] = $header_image;
            $data['main_image'] = $image;

            $page = Page::create($data);

            $this->insertTags($data['tags'], $page);

            if ($request->hasFile('gallery')) {
                foreach ($request->file('gallery') as $file) {
                    $image = $file->store('pages', 'public');
                    $page->images()->create([
                        'image' => $image,
                    ]);
                }
            }

            DB::commit();

        } catch (Throwable $e) {
            DB::rollBack();

            return $e;
        }

        return redirect()->route('pages.index')->with('success', __('Page ') . $page->name . __(' Created Done!'));
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function show(Page $page)
    {
        return view('control-panel.pages.show', [
            'page' => $page,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Page $page)
    {
        $tags = $page->tags()->pluck('name')->toArray();
        $tags = implode(', ', $tags);

        return view('control-panel.pages.edit', [
            'page' => $page,
            'tags' => $tags,
            'menus' => Menu::where('status','<>','2')->where('id','<>',2)->get()
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdatePageRequest $request, Page $page)
    {

        $data = $request->validated();

        $image = $page->main_image;
        $header_image = $page->header_image;

        DB::beginTransaction();

        try {

            if ($request->hasFile('main_image') && $request->file('main_image')->isValid()) {
                Storage::disk('public')->delete($image);
                $image = $request->file('main_image')->store('pages', 'public');
                $data['main_image'] = $image;
            }

            if ($request->hasFile('header_image') && $request->file('header_image')->isValid()) {
                Storage::disk('public')->delete($header_image);
                $header_image = $request->file('header_image')->store('pages', 'public');
                $data['header_image'] = $header_image;
            }


            foreach ($page->images as $gallery) {
                if ($request->has("check_" . $gallery->id) == 1) {
                    Storage::disk('public')->delete($gallery->image);
                    $gallery->delete();
                }
            }


            $page->update($data);
            $this->insertTags($data['tags'], $page);
            if ($request->hasFile('gallery')) {
                foreach ($request->file('gallery') as $file) {
                    $gallery_image = $file->store('pages', 'public');
                    $page->images()->create([
                        'image' => $gallery_image,
                    ]);
                }
            }
            DB::commit();

        } catch (Throwable $e) {
            DB::rollBack();
            return redirect()->route('pages.index')
                ->with('error', 'Operation failed');
        }

        return redirect()->route('pages.index')->with('success', __('Page ') . $page->name . __(' Updated Done!'));

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Page $page)
    {
        foreach ($page->images as $gallery) {
            Storage::disk('public')->delete($gallery->image);
            $gallery->delete();
        }

        Storage::disk('public')->delete($page->main_image);
        $page->delete();
        return redirect()->route('pages.index')->with('success', __('Page Deleted Done!'));
    }

    protected function insertTags($tags, $page)
    {
        PageTag::where('page_id', $page->id)->delete();
        $tags_ids = [];
        if ($tags) {
            $tags_array = explode(',', $tags);
            foreach ($tags_array as $tag_name) {
                $tag_name = trim($tag_name);
                $tag = Tag::where('name', $tag_name)->first();
                if (!$tag) {
                    $tag = Tag::create([
                        'name' => $tag_name,
                    ]);
                }
                PageTag::create([
                    'page_id' => $page->id,
                    'tag_id' => $tag->id,
                ]);
            }
        }

    }

    public function getSubMenus($subid = 0)
    {
        // $subMenu = SubMenu::where('menu_id', $subid)->get();
//        dd($subMenu);
//        $a = "";
//        foreach($subMenu as $s){
//            $a.="<option value='".$s->id."'>".$s->name."</option>";
//        }
        // return response()->json($subMenu);
    }
}
